#!/usr/bin/env python3
"""
Alan Watts TTS Bridge for SynergyX CrossPlatform
=================================================
Uses F5-TTS with voice cloning for Alan Watts / Joker voices.
Communicates with C# via JSON over stdin/stdout.
"""
import sys
import json
import os
import tempfile
import threading
import queue

# Suppress warnings for cleaner output
import warnings
warnings.filterwarnings('ignore')

class AlanWattsTTSBridge:
    def __init__(self):
        self.engine = None
        self.is_initialized = False
        self.current_voice = "alanwatts"
        self.speed = 0.95
        self.nfe_steps = 32
        self.audio_player = None
        self.speech_queue = queue.Queue()
        self.is_speaking = False
        self.volume = 1.0  # Volume 0.0-1.0
        
        # Voice reference configurations
        self.voice_configs = {
            "alanwatts": {
                "reference": "AlanReference.wav",
                "text": "and if im talking to you right now and you are doing this I want you to listen to the sound of my voice just as if it were noise dont try to make",
                "speed": 0.95,
                "nfe_steps": 32
            },
            "joker": {
                "reference": "JokerReference.wav",
                "text": "you wanna know how I got these scars",
                "speed": 1.05,
                "nfe_steps": 28
            },
            "cybertiger": {
                "reference": "CyberTigerReference.wav",
                "text": "Welcome to SynergyX. Your quantum wallet awaits.",
                "speed": 1.0,
                "nfe_steps": 24
            }
        }
        
    def send_response(self, response_type, **kwargs):
        """Send JSON response to C#"""
        response = {"type": response_type, **kwargs}
        print(json.dumps(response), flush=True)
    
    def send_status(self, message):
        self.send_response("status", message=message)
    
    def send_error(self, message):
        self.send_response("error", message=message)
    
    def find_reference_audio(self, filename):
        """Search for reference audio file"""
        search_paths = [
            os.path.dirname(__file__),
            os.path.join(os.path.dirname(__file__), "Tests"),
            os.path.join(os.path.dirname(__file__), "..", "SynxwalletAlpha", "Tests"),
            os.path.join(os.path.dirname(__file__), "..", "TTS", "F5TTS"),
            "C:/Users/venge/Desktop/SynergyX/SynxwalletAlpha/Tests",
        ]
        
        for path in search_paths:
            full_path = os.path.join(path, filename)
            if os.path.exists(full_path):
                return full_path
        
        return None
    
    def initialize(self, voice="alanwatts", speed=0.95, nfe_steps=32):
        """Initialize F5-TTS engine"""
        try:
            self.send_status("Loading F5-TTS model...")
            
            # Change to F5TTS directory for imports
            f5_path = os.path.join(os.path.dirname(__file__), "..", "TTS", "F5TTS")
            if os.path.exists(f5_path):
                sys.path.insert(0, f5_path)
                original_cwd = os.getcwd()
                os.chdir(f5_path)
            else:
                # Try alternate path
                f5_path = "C:/Users/venge/Desktop/SynergyX/TTS/F5TTS"
                if os.path.exists(f5_path):
                    sys.path.insert(0, f5_path)
                    original_cwd = os.getcwd()
                    os.chdir(f5_path)
                else:
                    self.send_error("F5-TTS directory not found")
                    return False
            
            try:
                from f5tts_pure_core import F5TTSPureEngine
                
                self.engine = F5TTSPureEngine()
                self.send_status("Initializing F5-TTS core...")
                
                if not self.engine.initialize("F5-TTS"):
                    self.send_error("Failed to initialize F5-TTS model")
                    return False
                
                # Set reference audio for selected voice
                self.current_voice = voice.lower()
                self.speed = speed
                self.nfe_steps = nfe_steps
                
                config = self.voice_configs.get(self.current_voice, self.voice_configs["alanwatts"])
                ref_path = self.find_reference_audio(config["reference"])
                
                if ref_path and os.path.exists(ref_path):
                    self.send_status(f"Loading voice: {self.current_voice}")
                    self.engine.set_reference(ref_path, config["text"])
                else:
                    self.send_status(f"Reference audio not found: {config['reference']}")
                    # Use fallback or continue without reference
                
                self.is_initialized = True
                self.send_status("🎙️ Alan Watts voice engine ready!")
                self.send_response("initialized", voice=self.current_voice)
                
                return True
                
            finally:
                os.chdir(original_cwd)
            
        except ImportError as e:
            self.send_error(f"F5-TTS import failed: {str(e)}")
            return False
        except Exception as e:
            self.send_error(f"Initialization error: {str(e)}")
            import traceback
            sys.stderr.write(traceback.format_exc())
            return False
    
    def speak(self, text, voice=None, speed=None, nfe_steps=None):
        """Synthesize and play speech"""
        if not self.is_initialized:
            if not self.initialize():
                return False
        
        try:
            self.is_speaking = True
            
            # Use provided params or defaults
            voice = voice or self.current_voice
            speed = speed or self.speed
            nfe_steps = nfe_steps or self.nfe_steps
            
            # Change voice if different
            if voice != self.current_voice:
                self.set_voice(voice)
            
            self.send_status(f"Synthesizing...")
            
            # Generate temp file
            temp_path = tempfile.mktemp(suffix=".wav", prefix="alan_speech_")
            
            # Change to F5TTS directory
            f5_path = os.path.join(os.path.dirname(__file__), "..", "TTS", "F5TTS")
            if not os.path.exists(f5_path):
                f5_path = "C:/Users/venge/Desktop/SynergyX/TTS/F5TTS"
            
            original_cwd = os.getcwd()
            os.chdir(f5_path)
            
            try:
                success = self.engine.synthesize_to_file(text, temp_path, speed=speed, nfe_step=nfe_steps)
            finally:
                os.chdir(original_cwd)
            
            if success and os.path.exists(temp_path):
                self.send_status("Playing audio...")
                
                # Play audio using platform-appropriate method
                self.play_audio(temp_path)
                
                # Cleanup temp file after playing
                try:
                    os.remove(temp_path)
                except:
                    pass
                
                self.send_response("speech_complete")
            else:
                self.send_error("Synthesis failed")
            
            self.is_speaking = False
            return True
            
        except Exception as e:
            self.send_error(f"Speech error: {str(e)}")
            self.is_speaking = False
            return False
    
    def synthesize(self, text, output_path, voice=None, speed=None, nfe_steps=None):
        """Synthesize text to file without playing"""
        if not self.is_initialized:
            if not self.initialize():
                return False
        
        try:
            voice = voice or self.current_voice
            speed = speed or self.speed
            nfe_steps = nfe_steps or self.nfe_steps
            
            if voice != self.current_voice:
                self.set_voice(voice)
            
            self.send_status(f"Synthesizing to file...")
            
            # Change to F5TTS directory
            f5_path = os.path.join(os.path.dirname(__file__), "..", "TTS", "F5TTS")
            if not os.path.exists(f5_path):
                f5_path = "C:/Users/venge/Desktop/SynergyX/TTS/F5TTS"
            
            original_cwd = os.getcwd()
            os.chdir(f5_path)
            
            try:
                success = self.engine.synthesize_to_file(text, output_path, speed=speed, nfe_step=nfe_steps)
            finally:
                os.chdir(original_cwd)
            
            if success:
                self.send_response("synthesized", output_path=output_path)
            else:
                self.send_error("Synthesis failed")
            
            return success
            
        except Exception as e:
            self.send_error(f"Synthesis error: {str(e)}")
            return False
    
    def set_voice(self, voice):
        """Change the voice/persona"""
        voice = voice.lower()
        if voice not in self.voice_configs:
            self.send_error(f"Unknown voice: {voice}")
            return False
        
        self.current_voice = voice
        config = self.voice_configs[voice]
        
        ref_path = self.find_reference_audio(config["reference"])
        if ref_path and self.engine:
            self.engine.set_reference(ref_path, config["text"])
            self.speed = config.get("speed", 0.95)
            self.nfe_steps = config.get("nfe_steps", 32)
            self.send_status(f"Voice changed to: {voice}")
            return True
        else:
            self.send_error(f"Reference audio not found for voice: {voice}")
            return False
    
    def play_audio(self, audio_path, volume=None):
        """Play audio file using platform-appropriate method with volume control"""
        try:
            import platform
            
            # Use provided volume or instance volume
            vol = volume if volume is not None else self.volume
            vol = max(0.0, min(1.0, vol))  # Clamp 0-1
            
            if platform.system() == "Windows":
                # Use pygame for Windows with volume support
                try:
                    import pygame
                    pygame.mixer.init()
                    pygame.mixer.music.load(audio_path)
                    pygame.mixer.music.set_volume(vol)  # Set volume 0.0-1.0
                    pygame.mixer.music.play()
                    while pygame.mixer.music.get_busy():
                        pygame.time.wait(100)
                    pygame.mixer.quit()
                except ImportError:
                    # Fallback to winsound (no volume control)
                    try:
                        import winsound
                        winsound.PlaySound(audio_path, winsound.SND_FILENAME)
                    except:
                        os.system(f'start /wait "" "{audio_path}"')
            
            elif platform.system() == "Darwin":  # macOS
                # afplay supports volume with -v flag (0-255)
                mac_vol = int(vol * 255)
                os.system(f'afplay -v {mac_vol} "{audio_path}"')
            
            else:  # Linux
                # Use paplay with volume (0-65536) or aplay
                linux_vol = int(vol * 65536)
                os.system(f'paplay --volume={linux_vol} "{audio_path}" 2>/dev/null || aplay "{audio_path}" 2>/dev/null')
            
        except Exception as e:
            sys.stderr.write(f"Audio playback error: {e}\n")
    
    def set_volume(self, volume):
        """Set the volume for TTS playback (0.0 to 1.0)"""
        self.volume = max(0.0, min(1.0, float(volume)))
        self.send_status(f"Volume set to {int(self.volume * 100)}%")
    
    def stop(self):
        """Stop current speech"""
        self.is_speaking = False
        # Kill any playing audio
        import platform
        if platform.system() == "Windows":
            os.system("taskkill /F /IM wmplayer.exe 2>nul")
        self.send_status("Speech stopped")
    
    def run(self):
        """Main loop - read commands from stdin"""
        self.send_status("Alan Watts TTS bridge starting...")
        
        while True:
            try:
                line = sys.stdin.readline()
                if not line:
                    break
                
                line = line.strip()
                if not line:
                    continue
                
                try:
                    cmd = json.loads(line)
                except json.JSONDecodeError:
                    self.send_error(f"Invalid JSON: {line}")
                    continue
                
                command = cmd.get("command", "")
                
                if command == "init":
                    voice = cmd.get("voice", "alanwatts")
                    speed = cmd.get("speed", 0.95)
                    nfe_steps = cmd.get("nfe_steps", 32)
                    self.initialize(voice, speed, nfe_steps)
                
                elif command == "speak":
                    text = cmd.get("text", "")
                    voice = cmd.get("voice")
                    speed = cmd.get("speed")
                    nfe_steps = cmd.get("nfe_steps")
                    volume = cmd.get("volume")
                    if volume is not None:
                        self.volume = max(0.0, min(1.0, float(volume)))
                    threading.Thread(target=self.speak, args=(text, voice, speed, nfe_steps)).start()
                
                elif command == "synthesize":
                    text = cmd.get("text", "")
                    output_path = cmd.get("output_path", "output.wav")
                    voice = cmd.get("voice")
                    speed = cmd.get("speed")
                    nfe_steps = cmd.get("nfe_steps")
                    self.synthesize(text, output_path, voice, speed, nfe_steps)
                
                elif command == "set_voice":
                    voice = cmd.get("voice", "alanwatts")
                    self.set_voice(voice)
                
                elif command == "set_volume":
                    volume = cmd.get("volume", 1.0)
                    self.set_volume(volume)
                
                elif command == "stop":
                    self.stop()
                
                elif command == "shutdown":
                    self.send_status("Shutting down...")
                    break
                
                else:
                    self.send_error(f"Unknown command: {command}")
            
            except Exception as e:
                self.send_error(f"Command error: {str(e)}")


if __name__ == "__main__":
    bridge = AlanWattsTTSBridge()
    bridge.run()
